module net.BurtonRadons.spyl.valueFloat;

private import net.BurtonRadons.spyl.value;
private import net.BurtonRadons.spyl.valueBoolean;
private import net.BurtonRadons.spyl.valueFunction;
private import net.BurtonRadons.spyl.valueInteger;
private import net.BurtonRadons.spyl.valueType;
private import std.c.stdio;
private import std.math, std.math2;

class FloatType : TypeValue
{
    private static FloatType singleton;
    private static bit locked = true;
    
    this ()
    {
        assert (!locked);
    }
    
    static FloatType create ()
    {
        if (singleton === null)
        {
            locked = false;
            singleton = new FloatType;
            locked = true;
        }
        return singleton;
    }
    
    char [] doc ()
    {
        return "A floating-point type.  The type has the constants "
            "infinity and nan.  Float "
            "constants can be specified by such numbers as 2.4 or 2e4.";
    }
    
    Value call (Value [] args)
    {
        Value result;
        
        assert (args.length == 1);
        result = args [0].castTo (this);
        assert ((FloatValue) result);
        return result;
    }
    
    override Value getattr (char [] name)
    {
        switch (name)
        {
            case "infinity": return new AttributeFunctionValue (&infinity, "This constant is positive infinity.");
            case "nan": return new AttributeFunctionValue (&nan, "This is a not-a-number constant.");
            default: return super.getattr (name);
        }
    }
    
    Value epsilon () { return new FloatValue (real.epsilon); }
    Value infinity () { return new FloatValue (real.infinity); }
    Value nan () { return new FloatValue (real.nan); }
}

class FloatValue : Value
{
    real content; /**< Actual value. */
    
    /** Assign the parameter. */
    this (real content)
    {
        this.content = content;
    }
    
    override char [] typeName ()
    {
        return "Float";
    }
    
    override char [] repr ()
    {
        char [640] buffer;
        int length;
        
        length = sprintf (buffer, "%Lg", content);
        return buffer [0 .. length].dup;
    }
    
    override Value getattr (char [] name)
    {
        switch (name)
        {
            case "isfinite": return new AttributeFunctionValue (&isfinite, "True if the value is not infinite.");
            case "isinfinite": return new AttributeFunctionValue (&isinfinite, "True if the value is infinite.");
            case "isnan": return new AttributeFunctionValue (&isnan, "True if the value is not a number.");
            case "sqrt": return new AttributeFunctionValue (&sqrt, "The square root of the value.");
            default: return super.getattr (name);
        }
    }

    Value isfinite () { return BooleanValue.create ((bit) math.isfinite (content)); }
    Value isinfinite () { return BooleanValue.create ((bit) !math.isfinite (content)); }
    Value isnan () { return BooleanValue.create ((bit) math2.isnan (content)); }    
    Value sqrt () { return new FloatValue (math2.sqrt (content)); }
    
    override Value opPow (Value other)
    {
        real exponent;
        
        if ((IntegerValue) other)
            exponent = ((IntegerValue) other).content;
        else if ((FloatValue) other)
            exponent = ((FloatValue) other).content;
        else
            return super.pow (other);
        
        return new FloatValue (math2.pow (content, exponent));
    }
    
    override Value opPow_r (Value other)
    {
        if ((IntegerValue) other)
            return new FloatValue (math2.pow (((IntegerValue) other).content, content));
        return super.pow_r (other);
    }
    
    override Value castTo (Value type)
    {
        if ((FloatType) type)
            return this;
        if ((IntegerType) type)
        {
            if (!math.isfinite (content) || math2.isnan (content))
                throw new Error ("Cannot cast non-normal floats to integer.");
            return new IntegerValue (content + 0.5);
        }
        return super.castTo (type);
    }
}
